<?php

namespace App\Models;

use CodeIgniter\Model;

class ProductModel extends Model
{
    protected $table = 'products';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'seller_id', 'brand_id', 'listing_type', 'gender', 'product_type', 'title', 
        'description', 'category', 'sub_category', 'size', 'color', 'used_times', 
        'original_price', 'suggested_sale_price', 'suggested_rental_cost', 'price', 
        'rental_cost', 'rental_deposit', 'fitting_charge', 'allow_alter_fitting',
        'has_bill', 'bill_image', 'dispatch_address', 'dispatch_pin_code',
        'rental_start_date', 'rental_end_date', 'status', 'admin_remarks', 'views_count'
    ];

    protected bool $allowEmptyInserts = false;
    protected bool $updateOnlyChanged = true;

    protected array $casts = [];
    protected array $castHandlers = [];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    protected $validationRules = [];
    
    // Removed strict validation - controller handles it

    protected $validationMessages = [];
    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    protected $allowCallbacks = true;
    protected $beforeInsert = [];
    protected $afterInsert = [];
    protected $beforeUpdate = [];
    protected $afterUpdate = [];
    protected $beforeFind = [];
    protected $afterFind = [];
    protected $beforeDelete = [];
    protected $afterDelete = [];

    public function getProductWithImages($productId)
    {
        return $this->db->table('products p')
            ->select('p.*, u.name as seller_name, u.email as seller_email, u.reliability_score')
            ->select('GROUP_CONCAT(pi.image_path) as images')
            ->join('users u', 'u.id = p.seller_id')
            ->join('product_images pi', 'pi.product_id = p.id', 'left')
            ->where('p.id', $productId)
            ->groupBy('p.id')
            ->get()
            ->getRowArray();
    }

    public function getSellerProducts($sellerId, $status = null)
    {
        $builder = $this->db->table('products p')
            ->select('p.*, COUNT(DISTINCT o.id) as offer_count, COUNT(DISTINCT pi.id) as image_count')
            ->join('offers o', 'o.product_id = p.id AND o.status = "pending"', 'left')
            ->join('product_images pi', 'pi.product_id = p.id', 'left')
            ->where('p.seller_id', $sellerId)
            ->groupBy('p.id')
            ->orderBy('p.created_at', 'DESC');

        if ($status) {
            $builder->where('p.status', $status);
        }

        return $builder->get()->getResultArray();
    }

    public function getPendingApprovals()
    {
        return $this->db->table('products p')
            ->select('p.*, u.name as seller_name, u.email as seller_email, u.mobile, COUNT(pi.id) as image_count')
            ->join('users u', 'u.id = p.seller_id')
            ->join('product_images pi', 'pi.product_id = p.id', 'left')
            ->where('p.status', 'pending')
            ->groupBy('p.id')
            ->orderBy('p.created_at', 'ASC')
            ->get()
            ->getResultArray();
    }

    public function approveProduct($productId, $remarks = null)
    {
        return $this->update($productId, [
            'status' => 'approved',
            'admin_remarks' => $remarks
        ]);
    }

    public function rejectProduct($productId, $remarks)
    {
        return $this->update($productId, [
            'status' => 'rejected',
            'admin_remarks' => $remarks
        ]);
    }

    public function incrementViews($productId)
    {
        $this->db->query("UPDATE products SET views_count = views_count + 1 WHERE id = ?", [$productId]);
    }
}
